/*
 * Decompiled with CFR 0.152.
 */
package net.runelite.client.plugins.loottracker;

import com.google.common.collect.HashMultiset;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multiset;
import com.google.common.collect.Multisets;
import com.google.inject.Provides;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.swing.SwingUtilities;
import net.runelite.api.ChatMessageType;
import net.runelite.api.Client;
import net.runelite.api.InventoryID;
import net.runelite.api.ItemComposition;
import net.runelite.api.ItemContainer;
import net.runelite.api.NPC;
import net.runelite.api.Player;
import net.runelite.api.coords.WorldPoint;
import net.runelite.api.events.ChatMessage;
import net.runelite.api.events.ConfigChanged;
import net.runelite.api.events.ItemContainerChanged;
import net.runelite.api.events.WidgetLoaded;
import net.runelite.client.account.AccountSession;
import net.runelite.client.account.SessionManager;
import net.runelite.client.callback.ClientThread;
import net.runelite.client.config.ConfigManager;
import net.runelite.client.eventbus.Subscribe;
import net.runelite.client.events.NpcLootReceived;
import net.runelite.client.events.PlayerLootReceived;
import net.runelite.client.events.SessionClose;
import net.runelite.client.events.SessionOpen;
import net.runelite.client.game.ItemManager;
import net.runelite.client.game.ItemStack;
import net.runelite.client.game.SpriteManager;
import net.runelite.client.plugins.Plugin;
import net.runelite.client.plugins.PluginDescriptor;
import net.runelite.client.plugins.loottracker.LootTrackerConfig;
import net.runelite.client.plugins.loottracker.LootTrackerItem;
import net.runelite.client.plugins.loottracker.LootTrackerPanel;
import net.runelite.client.plugins.loottracker.LootTrackerRecord;
import net.runelite.client.ui.ClientToolbar;
import net.runelite.client.ui.NavigationButton;
import net.runelite.client.util.ImageUtil;
import net.runelite.client.util.Text;
import net.runelite.http.api.loottracker.GameItem;
import net.runelite.http.api.loottracker.LootRecord;
import net.runelite.http.api.loottracker.LootRecordType;
import net.runelite.http.api.loottracker.LootTrackerClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PluginDescriptor(name="Loot Tracker", description="Tracks loot from monsters and minigames", tags={"drops"}, enabledByDefault=false)
public class LootTrackerPlugin
extends Plugin {
    private static final Logger log = LoggerFactory.getLogger(LootTrackerPlugin.class);
    private static final Pattern CLUE_SCROLL_PATTERN = Pattern.compile("You have completed [0-9]+ ([a-z]+) Treasure Trails.");
    private static final int THEATRE_OF_BLOOD_REGION = 12867;
    private static final String HERBIBOAR_LOOTED_MESSAGE = "You harvest herbs from the herbiboar, whereupon it escapes.";
    private static final String HERBIBOR_EVENT = "Herbiboar";
    private static final String CHEST_LOOTED_MESSAGE = "You find some treasure in the chest!";
    private static final Map<Integer, String> CHEST_EVENT_TYPES = ImmutableMap.of(5179, "Brimstone Chest", 11573, "Crystal Chest");
    @Inject
    private ClientToolbar clientToolbar;
    @Inject
    private ItemManager itemManager;
    @Inject
    private SpriteManager spriteManager;
    @Inject
    private LootTrackerConfig config;
    @Inject
    private Client client;
    @Inject
    private ClientThread clientThread;
    @Inject
    private SessionManager sessionManager;
    @Inject
    private ScheduledExecutorService executor;
    private LootTrackerPanel panel;
    private NavigationButton navButton;
    private String eventType;
    private List<String> ignoredItems = new ArrayList<String>();
    private Multiset<Integer> inventorySnapshot;
    private LootTrackerClient lootTrackerClient;

    private static Collection<ItemStack> stack(Collection<ItemStack> items) {
        ArrayList<ItemStack> list = new ArrayList<ItemStack>();
        for (ItemStack item : items) {
            int quantity = 0;
            for (ItemStack i : list) {
                if (i.getId() != item.getId()) continue;
                quantity = i.getQuantity();
                list.remove(i);
                break;
            }
            if (quantity > 0) {
                list.add(new ItemStack(item.getId(), item.getQuantity() + quantity, item.getLocation()));
                continue;
            }
            list.add(item);
        }
        return list;
    }

    @Provides
    LootTrackerConfig provideConfig(ConfigManager configManager) {
        return configManager.getConfig(LootTrackerConfig.class);
    }

    @Subscribe
    public void onSessionOpen(SessionOpen sessionOpen) {
        AccountSession accountSession = this.sessionManager.getAccountSession();
        this.lootTrackerClient = accountSession.getUuid() != null ? new LootTrackerClient(accountSession.getUuid()) : null;
    }

    @Subscribe
    public void onSessionClose(SessionClose sessionClose) {
        this.lootTrackerClient = null;
    }

    @Subscribe
    public void onConfigChanged(ConfigChanged event) {
        if (event.getGroup().equals("loottracker")) {
            this.ignoredItems = Text.fromCSV(this.config.getIgnoredItems());
            SwingUtilities.invokeLater(this.panel::updateIgnoredRecords);
        }
    }

    @Override
    protected void startUp() throws Exception {
        this.ignoredItems = Text.fromCSV(this.config.getIgnoredItems());
        this.panel = new LootTrackerPanel(this, this.itemManager, this.config);
        this.spriteManager.getSpriteAsync(900, 0, this.panel::loadHeaderIcon);
        BufferedImage icon = ImageUtil.getResourceStreamFromClass(this.getClass(), "panel_icon.png");
        this.navButton = NavigationButton.builder().tooltip("Loot Tracker").icon(icon).priority(5).panel(this.panel).build();
        this.clientToolbar.addNavigation(this.navButton);
        AccountSession accountSession = this.sessionManager.getAccountSession();
        if (accountSession != null) {
            this.lootTrackerClient = new LootTrackerClient(accountSession.getUuid());
            this.clientThread.invokeLater(() -> {
                switch (this.client.getGameState()) {
                    case STARTING: 
                    case UNKNOWN: {
                        return false;
                    }
                }
                this.executor.submit(() -> {
                    Collection<LootRecord> lootRecords;
                    if (!this.config.syncPanel()) {
                        return;
                    }
                    try {
                        lootRecords = this.lootTrackerClient.get();
                    }
                    catch (IOException e) {
                        log.debug("Unable to look up loot", e);
                        return;
                    }
                    log.debug("Loaded {} data entries", (Object)lootRecords.size());
                    this.clientThread.invokeLater(() -> {
                        Collection<LootTrackerRecord> records = this.convertToLootTrackerRecord(lootRecords);
                        SwingUtilities.invokeLater(() -> this.panel.addRecords(records));
                    });
                });
                return true;
            });
        }
    }

    @Override
    protected void shutDown() {
        this.clientToolbar.removeNavigation(this.navButton);
        this.lootTrackerClient = null;
    }

    @Subscribe
    public void onNpcLootReceived(NpcLootReceived npcLootReceived) {
        NPC npc = npcLootReceived.getNpc();
        Collection<ItemStack> items = npcLootReceived.getItems();
        String name = npc.getName();
        int combat = npc.getCombatLevel();
        LootTrackerItem[] entries = this.buildEntries(LootTrackerPlugin.stack(items));
        SwingUtilities.invokeLater(() -> this.panel.add(name, combat, entries));
        if (this.lootTrackerClient != null && this.config.saveLoot()) {
            LootRecord lootRecord = new LootRecord(name, LootRecordType.NPC, LootTrackerPlugin.toGameItems(items), Instant.now());
            this.lootTrackerClient.submit(lootRecord);
        }
    }

    @Subscribe
    public void onPlayerLootReceived(PlayerLootReceived playerLootReceived) {
        Player player = playerLootReceived.getPlayer();
        Collection<ItemStack> items = playerLootReceived.getItems();
        String name = player.getName();
        int combat = player.getCombatLevel();
        LootTrackerItem[] entries = this.buildEntries(LootTrackerPlugin.stack(items));
        SwingUtilities.invokeLater(() -> this.panel.add(name, combat, entries));
        if (this.lootTrackerClient != null && this.config.saveLoot()) {
            LootRecord lootRecord = new LootRecord(name, LootRecordType.PLAYER, LootTrackerPlugin.toGameItems(items), Instant.now());
            this.lootTrackerClient.submit(lootRecord);
        }
    }

    @Subscribe
    public void onWidgetLoaded(WidgetLoaded event) {
        ItemContainer container;
        switch (event.getGroupId()) {
            case 155: {
                this.eventType = "Barrows";
                container = this.client.getItemContainer(InventoryID.BARROWS_REWARD);
                break;
            }
            case 539: {
                this.eventType = "Chambers of Xeric";
                container = this.client.getItemContainer(InventoryID.CHAMBERS_OF_XERIC_CHEST);
                break;
            }
            case 23: {
                int region = WorldPoint.fromLocalInstance(this.client, this.client.getLocalPlayer().getLocalLocation()).getRegionID();
                if (region != 12867) {
                    return;
                }
                this.eventType = "Theatre of Blood";
                container = this.client.getItemContainer(InventoryID.THEATRE_OF_BLOOD_CHEST);
                break;
            }
            case 73: {
                container = this.client.getItemContainer(InventoryID.BARROWS_REWARD);
                break;
            }
            default: {
                return;
            }
        }
        if (container == null) {
            return;
        }
        Collection items = Arrays.stream(container.getItems()).filter(item -> item.getId() > 0).map(item -> new ItemStack(item.getId(), item.getQuantity(), this.client.getLocalPlayer().getLocalLocation())).collect(Collectors.toList());
        if (items.isEmpty()) {
            log.debug("No items to find for Event: {} | Container: {}", (Object)this.eventType, (Object)container);
            return;
        }
        LootTrackerItem[] entries = this.buildEntries(LootTrackerPlugin.stack(items));
        SwingUtilities.invokeLater(() -> this.panel.add(this.eventType, -1, entries));
        if (this.lootTrackerClient != null && this.config.saveLoot()) {
            LootRecord lootRecord = new LootRecord(this.eventType, LootRecordType.EVENT, LootTrackerPlugin.toGameItems(items), Instant.now());
            this.lootTrackerClient.submit(lootRecord);
        }
    }

    @Subscribe
    public void onChatMessage(ChatMessage event) {
        if (event.getType() != ChatMessageType.GAMEMESSAGE && event.getType() != ChatMessageType.SPAM) {
            return;
        }
        String message = event.getMessage();
        if (message.equals(CHEST_LOOTED_MESSAGE)) {
            int regionID = this.client.getLocalPlayer().getWorldLocation().getRegionID();
            if (!CHEST_EVENT_TYPES.containsKey(regionID)) {
                return;
            }
            this.eventType = CHEST_EVENT_TYPES.get(regionID);
            this.takeInventorySnapshot();
            return;
        }
        if (message.equals(HERBIBOAR_LOOTED_MESSAGE)) {
            this.eventType = HERBIBOR_EVENT;
            this.takeInventorySnapshot();
            return;
        }
        Matcher m = CLUE_SCROLL_PATTERN.matcher(Text.removeTags(message));
        if (m.find()) {
            String type;
            switch (type = m.group(1).toLowerCase()) {
                case "beginner": {
                    this.eventType = "Clue Scroll (Beginner)";
                    break;
                }
                case "easy": {
                    this.eventType = "Clue Scroll (Easy)";
                    break;
                }
                case "medium": {
                    this.eventType = "Clue Scroll (Medium)";
                    break;
                }
                case "hard": {
                    this.eventType = "Clue Scroll (Hard)";
                    break;
                }
                case "elite": {
                    this.eventType = "Clue Scroll (Elite)";
                    break;
                }
                case "master": {
                    this.eventType = "Clue Scroll (Master)";
                }
            }
        }
    }

    @Subscribe
    public void onItemContainerChanged(ItemContainerChanged event) {
        if (this.eventType != null && (CHEST_EVENT_TYPES.containsValue(this.eventType) || HERBIBOR_EVENT.equals(this.eventType))) {
            if (event.getItemContainer() != this.client.getItemContainer(InventoryID.INVENTORY)) {
                return;
            }
            this.processChestLoot(this.eventType, event.getItemContainer());
            this.eventType = null;
        }
    }

    private void takeInventorySnapshot() {
        ItemContainer itemContainer = this.client.getItemContainer(InventoryID.INVENTORY);
        if (itemContainer != null) {
            this.inventorySnapshot = HashMultiset.create();
            Arrays.stream(itemContainer.getItems()).forEach(item -> this.inventorySnapshot.add(item.getId(), item.getQuantity()));
        }
    }

    private void processChestLoot(String chestType, ItemContainer inventoryContainer) {
        if (this.inventorySnapshot != null) {
            HashMultiset currentInventory = HashMultiset.create();
            Arrays.stream(inventoryContainer.getItems()).forEach(item -> currentInventory.add(item.getId(), item.getQuantity()));
            Multiset diff = Multisets.difference(currentInventory, this.inventorySnapshot);
            List<ItemStack> items = diff.entrySet().stream().map(e -> new ItemStack((Integer)e.getElement(), e.getCount(), this.client.getLocalPlayer().getLocalLocation())).collect(Collectors.toList());
            LootTrackerItem[] entries = this.buildEntries(LootTrackerPlugin.stack(items));
            SwingUtilities.invokeLater(() -> this.panel.add(chestType, -1, entries));
            if (this.lootTrackerClient != null && this.config.saveLoot()) {
                LootRecord lootRecord = new LootRecord(chestType, LootRecordType.EVENT, LootTrackerPlugin.toGameItems(items), Instant.now());
                this.lootTrackerClient.submit(lootRecord);
            }
            this.inventorySnapshot = null;
        }
    }

    void toggleItem(String name, boolean ignore) {
        HashSet<String> ignoredItemSet = new HashSet<String>(this.ignoredItems);
        if (ignore) {
            ignoredItemSet.add(name);
        } else {
            ignoredItemSet.remove(name);
        }
        this.config.setIgnoredItems(Text.toCSV(ignoredItemSet));
        this.panel.updateIgnoredRecords();
    }

    boolean isIgnored(String name) {
        return this.ignoredItems.contains(name);
    }

    private LootTrackerItem buildLootTrackerItem(int itemId, int quantity) {
        ItemComposition itemComposition = this.itemManager.getItemComposition(itemId);
        int realItemId = itemComposition.getNote() != -1 ? itemComposition.getLinkedNoteId() : itemId;
        long price = (long)this.itemManager.getItemPrice(realItemId) * (long)quantity;
        boolean ignored = this.ignoredItems.contains(itemComposition.getName());
        return new LootTrackerItem(itemId, itemComposition.getName(), quantity, price, ignored);
    }

    private LootTrackerItem[] buildEntries(Collection<ItemStack> itemStacks) {
        return (LootTrackerItem[])itemStacks.stream().map(itemStack -> this.buildLootTrackerItem(itemStack.getId(), itemStack.getQuantity())).toArray(LootTrackerItem[]::new);
    }

    private static Collection<GameItem> toGameItems(Collection<ItemStack> items) {
        return items.stream().map(item -> new GameItem(item.getId(), item.getQuantity())).collect(Collectors.toList());
    }

    private Collection<LootTrackerRecord> convertToLootTrackerRecord(Collection<LootRecord> records) {
        ArrayList<LootTrackerRecord> trackerRecords = new ArrayList<LootTrackerRecord>();
        for (LootRecord record : records) {
            LootTrackerItem[] drops = (LootTrackerItem[])record.getDrops().stream().map(itemStack -> this.buildLootTrackerItem(itemStack.getId(), itemStack.getQty())).toArray(LootTrackerItem[]::new);
            trackerRecords.add(new LootTrackerRecord(record.getEventId(), "", drops, -1L));
        }
        return trackerRecords;
    }

    LootTrackerClient getLootTrackerClient() {
        return this.lootTrackerClient;
    }
}

