/*
 * Decompiled with CFR 0.152.
 */
package net.runelite.client.config;

import com.google.common.base.Strings;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableMap;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Proxy;
import java.nio.channels.FileLock;
import java.nio.charset.Charset;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Singleton;
import net.runelite.api.coords.WorldPoint;
import net.runelite.api.events.ConfigChanged;
import net.runelite.client.RuneLite;
import net.runelite.client.account.AccountSession;
import net.runelite.client.config.Alpha;
import net.runelite.client.config.ConfigDescriptor;
import net.runelite.client.config.ConfigGroup;
import net.runelite.client.config.ConfigInvocationHandler;
import net.runelite.client.config.ConfigItem;
import net.runelite.client.config.ConfigItemDescriptor;
import net.runelite.client.config.Keybind;
import net.runelite.client.config.ModifierlessKeybind;
import net.runelite.client.config.Range;
import net.runelite.client.eventbus.EventBus;
import net.runelite.client.util.ColorUtil;
import net.runelite.http.api.config.ConfigClient;
import net.runelite.http.api.config.ConfigEntry;
import net.runelite.http.api.config.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class ConfigManager {
    private static final Logger log = LoggerFactory.getLogger(ConfigManager.class);
    private static final String SETTINGS_FILE_NAME = "settings.properties";
    private static final DateFormat TIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss");
    @Inject
    EventBus eventBus;
    private final ScheduledExecutorService executor;
    private AccountSession session;
    private ConfigClient client;
    private File propertiesFile;
    private final ConfigInvocationHandler handler = new ConfigInvocationHandler(this);
    private final Properties properties = new Properties();
    private final Map<String, String> pendingChanges = new HashMap<String, String>();

    @Inject
    public ConfigManager(ScheduledExecutorService scheduledExecutorService) {
        this.executor = scheduledExecutorService;
        this.propertiesFile = this.getPropertiesFile();
        this.executor.scheduleWithFixedDelay(this::sendConfig, 30L, 30L, TimeUnit.SECONDS);
    }

    public final void switchSession(AccountSession session) {
        this.sendConfig();
        if (session == null) {
            this.session = null;
            this.client = null;
        } else {
            this.session = session;
            this.client = new ConfigClient(session.getUuid());
        }
        this.propertiesFile = this.getPropertiesFile();
        this.load();
    }

    private File getLocalPropertiesFile() {
        return new File(RuneLite.RUNELITE_DIR, SETTINGS_FILE_NAME);
    }

    private File getPropertiesFile() {
        if (this.session == null || this.session.getUsername() == null) {
            return this.getLocalPropertiesFile();
        }
        File profileDir = new File(RuneLite.PROFILES_DIR, this.session.getUsername().toLowerCase());
        return new File(profileDir, SETTINGS_FILE_NAME);
    }

    public void load() {
        Configuration configuration;
        if (this.client == null) {
            this.loadFromFile();
            return;
        }
        try {
            configuration = this.client.get();
        }
        catch (IOException ex) {
            log.debug("Unable to load configuration from client, using saved configuration from disk", ex);
            this.loadFromFile();
            return;
        }
        if (configuration.getConfig() == null || configuration.getConfig().isEmpty()) {
            log.debug("No configuration from client, using saved configuration on disk");
            this.loadFromFile();
            return;
        }
        this.properties.clear();
        for (ConfigEntry entry : configuration.getConfig()) {
            log.debug("Loading configuration value from client {}: {}", (Object)entry.getKey(), (Object)entry.getValue());
            String[] split = entry.getKey().split("\\.", 2);
            if (split.length != 2) continue;
            String groupName = split[0];
            String key = split[1];
            String value = entry.getValue();
            String oldValue = (String)this.properties.setProperty(entry.getKey(), value);
            ConfigChanged configChanged = new ConfigChanged();
            configChanged.setGroup(groupName);
            configChanged.setKey(key);
            configChanged.setOldValue(oldValue);
            configChanged.setNewValue(value);
            this.eventBus.post(configChanged);
        }
        try {
            this.saveToFile(this.propertiesFile);
            log.debug("Updated configuration on disk with the latest version");
        }
        catch (IOException ex) {
            log.warn("Unable to update configuration on disk", ex);
        }
    }

    private synchronized void syncPropertiesFromFile(File propertiesFile) {
        Properties properties = new Properties();
        try (FileInputStream in = new FileInputStream(propertiesFile);){
            properties.load(new InputStreamReader((InputStream)in, Charset.forName("UTF-8")));
        }
        catch (Exception e) {
            log.debug("Malformed properties, skipping update");
            return;
        }
        ImmutableMap<Object, Object> copy = ImmutableMap.copyOf(this.properties);
        copy.forEach((groupAndKey, value) -> {
            if (!properties.containsKey(groupAndKey)) {
                String[] split = groupAndKey.split("\\.", 2);
                if (split.length != 2) {
                    return;
                }
                String groupName = split[0];
                String key = split[1];
                this.unsetConfiguration(groupName, key);
            }
        });
        properties.forEach((BiConsumer<? super Object, ? super Object>)((BiConsumer<Object, Object>)(objGroupAndKey, objValue) -> {
            String groupAndKey = String.valueOf(objGroupAndKey);
            String[] split = groupAndKey.split("\\.", 2);
            if (split.length != 2) {
                return;
            }
            String groupName = split[0];
            String key = split[1];
            String value = String.valueOf(objValue);
            this.setConfiguration(groupName, key, value);
        }));
    }

    public void importLocal() {
        if (this.session == null) {
            return;
        }
        File file = new File(this.propertiesFile.getParent(), this.propertiesFile.getName() + "." + TIME_FORMAT.format(new Date()));
        try {
            this.saveToFile(file);
        }
        catch (IOException e) {
            log.warn("Backup failed, skipping import", e);
            return;
        }
        this.syncPropertiesFromFile(this.getLocalPropertiesFile());
    }

    private synchronized void loadFromFile() {
        this.properties.clear();
        try (FileInputStream in = new FileInputStream(this.propertiesFile);){
            this.properties.load(new InputStreamReader((InputStream)in, Charset.forName("UTF-8")));
        }
        catch (FileNotFoundException ex) {
            log.debug("Unable to load settings - no such file");
        }
        catch (IOException | IllegalArgumentException ex) {
            log.warn("Unable to load settings", ex);
        }
        try {
            ImmutableMap<Object, Object> copy = ImmutableMap.copyOf(this.properties);
            copy.forEach((groupAndKey, value) -> {
                String[] split = groupAndKey.split("\\.", 2);
                if (split.length != 2) {
                    log.debug("Properties key malformed!: {}", groupAndKey);
                    this.properties.remove(groupAndKey);
                    return;
                }
                String groupName = split[0];
                String key = split[1];
                ConfigChanged configChanged = new ConfigChanged();
                configChanged.setGroup(groupName);
                configChanged.setKey(key);
                configChanged.setOldValue(null);
                configChanged.setNewValue((String)value);
                this.eventBus.post(configChanged);
            });
        }
        catch (Exception ex) {
            log.warn("Error posting config events", ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void saveToFile(File propertiesFile) throws IOException {
        propertiesFile.getParentFile().mkdirs();
        try (FileOutputStream out = new FileOutputStream(propertiesFile);){
            FileLock lock = out.getChannel().lock();
            try {
                this.properties.store(new OutputStreamWriter((OutputStream)out, Charset.forName("UTF-8")), "RuneLite configuration");
            }
            finally {
                lock.release();
            }
        }
    }

    public <T> T getConfig(Class<T> clazz) {
        if (!Modifier.isPublic(clazz.getModifiers())) {
            throw new RuntimeException("Non-public configuration classes can't have default methods invoked");
        }
        Object t = Proxy.newProxyInstance(clazz.getClassLoader(), new Class[]{clazz}, (InvocationHandler)this.handler);
        return (T)t;
    }

    public List<String> getConfigurationKeys(String prefix) {
        return this.properties.keySet().stream().filter(v -> ((String)v).startsWith(prefix)).map(String.class::cast).collect(Collectors.toList());
    }

    public String getConfiguration(String groupName, String key) {
        return this.properties.getProperty(groupName + "." + key);
    }

    public <T> T getConfiguration(String groupName, String key, Class<T> clazz) {
        String value = this.getConfiguration(groupName, key);
        if (!Strings.isNullOrEmpty(value)) {
            try {
                return (T)ConfigManager.stringToObject(value, clazz);
            }
            catch (Exception e) {
                log.warn("Unable to unmarshal {}.{} ", groupName, key, e);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setConfiguration(String groupName, String key, String value) {
        String oldValue = (String)this.properties.setProperty(groupName + "." + key, value);
        if (Objects.equals(oldValue, value)) {
            return;
        }
        log.debug("Setting configuration value for {}.{} to {}", groupName, key, value);
        Map<String, String> map = this.pendingChanges;
        synchronized (map) {
            this.pendingChanges.put(groupName + "." + key, value);
        }
        ConfigChanged configChanged = new ConfigChanged();
        configChanged.setGroup(groupName);
        configChanged.setKey(key);
        configChanged.setOldValue(oldValue);
        configChanged.setNewValue(value);
        this.eventBus.post(configChanged);
    }

    public void setConfiguration(String groupName, String key, Object value) {
        this.setConfiguration(groupName, key, ConfigManager.objectToString(value));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void unsetConfiguration(String groupName, String key) {
        String oldValue = (String)this.properties.remove(groupName + "." + key);
        if (oldValue == null) {
            return;
        }
        log.debug("Unsetting configuration value for {}.{}", (Object)groupName, (Object)key);
        Map<String, String> map = this.pendingChanges;
        synchronized (map) {
            this.pendingChanges.put(groupName + "." + key, null);
        }
        ConfigChanged configChanged = new ConfigChanged();
        configChanged.setGroup(groupName);
        configChanged.setKey(key);
        configChanged.setOldValue(oldValue);
        this.eventBus.post(configChanged);
    }

    public ConfigDescriptor getConfigDescriptor(Object configurationProxy) {
        Class<?> inter = configurationProxy.getClass().getInterfaces()[0];
        ConfigGroup group = inter.getAnnotation(ConfigGroup.class);
        if (group == null) {
            throw new IllegalArgumentException("Not a config group");
        }
        List<ConfigItemDescriptor> items = Arrays.stream(inter.getMethods()).filter(m -> m.getParameterCount() == 0).map(m -> new ConfigItemDescriptor(m.getDeclaredAnnotation(ConfigItem.class), m.getReturnType(), m.getDeclaredAnnotation(Range.class), m.getDeclaredAnnotation(Alpha.class))).sorted((a, b) -> ComparisonChain.start().compare(a.getItem().position(), b.getItem().position()).compare((Comparable<?>)((Object)a.getItem().name()), (Comparable<?>)((Object)b.getItem().name())).result()).collect(Collectors.toList());
        return new ConfigDescriptor(group, items);
    }

    public void setDefaultConfiguration(Object proxy, boolean override) {
        Class<?> clazz = proxy.getClass().getInterfaces()[0];
        ConfigGroup group = clazz.getAnnotation(ConfigGroup.class);
        if (group == null) {
            return;
        }
        for (Method method : clazz.getDeclaredMethods()) {
            Object defaultValue;
            String current;
            ConfigItem item = method.getAnnotation(ConfigItem.class);
            if (item == null || method.getParameterCount() != 0) continue;
            if (!method.isDefault()) {
                if (!override || (current = this.getConfiguration(group.value(), item.keyName())) == null) continue;
                this.unsetConfiguration(group.value(), item.keyName());
                continue;
            }
            if (!override && (current = this.getConfiguration(group.value(), item.keyName())) != null) continue;
            try {
                defaultValue = ConfigInvocationHandler.callDefaultMethod(proxy, method, null);
            }
            catch (Throwable ex) {
                log.warn(null, ex);
                continue;
            }
            String current2 = this.getConfiguration(group.value(), item.keyName());
            String valueString = ConfigManager.objectToString(defaultValue);
            if (Objects.equals(current2, valueString)) continue;
            log.debug("Setting default configuration value for {}.{} to {}", group.value(), item.keyName(), defaultValue);
            this.setConfiguration(group.value(), item.keyName(), valueString);
        }
    }

    static Object stringToObject(String str, Class<?> type) {
        if (type == Boolean.TYPE || type == Boolean.class) {
            return Boolean.parseBoolean(str);
        }
        if (type == Integer.TYPE) {
            return Integer.parseInt(str);
        }
        if (type == Color.class) {
            return ColorUtil.fromString(str);
        }
        if (type == Dimension.class) {
            String[] splitStr = str.split("x");
            int width = Integer.parseInt(splitStr[0]);
            int height = Integer.parseInt(splitStr[1]);
            return new Dimension(width, height);
        }
        if (type == Point.class) {
            String[] splitStr = str.split(":");
            int width = Integer.parseInt(splitStr[0]);
            int height = Integer.parseInt(splitStr[1]);
            return new Point(width, height);
        }
        if (type == Rectangle.class) {
            String[] splitStr = str.split(":");
            int x = Integer.parseInt(splitStr[0]);
            int y = Integer.parseInt(splitStr[1]);
            int width = Integer.parseInt(splitStr[2]);
            int height = Integer.parseInt(splitStr[3]);
            return new Rectangle(x, y, width, height);
        }
        if (type.isEnum()) {
            return Enum.valueOf(type, str);
        }
        if (type == Instant.class) {
            return Instant.parse(str);
        }
        if (type == Keybind.class || type == ModifierlessKeybind.class) {
            String[] splitStr = str.split(":");
            int code = Integer.parseInt(splitStr[0]);
            int mods = Integer.parseInt(splitStr[1]);
            if (type == ModifierlessKeybind.class) {
                return new ModifierlessKeybind(code, mods);
            }
            return new Keybind(code, mods);
        }
        if (type == WorldPoint.class) {
            String[] splitStr = str.split(":");
            int x = Integer.parseInt(splitStr[0]);
            int y = Integer.parseInt(splitStr[1]);
            int plane = Integer.parseInt(splitStr[2]);
            return new WorldPoint(x, y, plane);
        }
        if (type == Duration.class) {
            return Duration.ofMillis(Long.parseLong(str));
        }
        return str;
    }

    static String objectToString(Object object) {
        if (object instanceof Color) {
            return String.valueOf(((Color)object).getRGB());
        }
        if (object instanceof Enum) {
            return ((Enum)object).name();
        }
        if (object instanceof Dimension) {
            Dimension d = (Dimension)object;
            return d.width + "x" + d.height;
        }
        if (object instanceof Point) {
            Point p = (Point)object;
            return p.x + ":" + p.y;
        }
        if (object instanceof Rectangle) {
            Rectangle r = (Rectangle)object;
            return r.x + ":" + r.y + ":" + r.width + ":" + r.height;
        }
        if (object instanceof Instant) {
            return ((Instant)object).toString();
        }
        if (object instanceof Keybind) {
            Keybind k = (Keybind)object;
            return k.getKeyCode() + ":" + k.getModifiers();
        }
        if (object instanceof WorldPoint) {
            WorldPoint wp = (WorldPoint)object;
            return wp.getX() + ":" + wp.getY() + ":" + wp.getPlane();
        }
        if (object instanceof Duration) {
            return Long.toString(((Duration)object).toMillis());
        }
        return object.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void sendConfig() {
        boolean changed;
        Map<String, String> map = this.pendingChanges;
        synchronized (map) {
            if (this.client != null) {
                for (Map.Entry<String, String> entry : this.pendingChanges.entrySet()) {
                    String key = entry.getKey();
                    String value = entry.getValue();
                    if (Strings.isNullOrEmpty(value)) {
                        this.client.unset(key);
                        continue;
                    }
                    this.client.set(key, value);
                }
            }
            changed = !this.pendingChanges.isEmpty();
            this.pendingChanges.clear();
        }
        if (changed) {
            try {
                this.saveToFile(this.propertiesFile);
            }
            catch (IOException ex) {
                log.warn("unable to save configuration file", ex);
            }
        }
    }
}

