/*
 * Decompiled with CFR 0.152.
 */
package net.runelite.client;

import com.google.common.base.Strings;
import com.google.common.escape.Escaper;
import com.google.common.escape.Escapers;
import com.google.inject.Inject;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.TrayIcon;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.inject.Singleton;
import net.runelite.api.ChatMessageType;
import net.runelite.api.Client;
import net.runelite.api.GameState;
import net.runelite.client.RuneLite;
import net.runelite.client.RuneLiteProperties;
import net.runelite.client.chat.ChatColorType;
import net.runelite.client.chat.ChatMessageBuilder;
import net.runelite.client.chat.ChatMessageManager;
import net.runelite.client.chat.QueuedMessage;
import net.runelite.client.config.RuneLiteConfig;
import net.runelite.client.ui.ClientUI;
import net.runelite.client.util.OSType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class Notifier {
    private static final Logger log = LoggerFactory.getLogger(Notifier.class);
    private static final int DEFAULT_TIMEOUT = 10000;
    private static final String DOUBLE_QUOTE = "\"";
    private static final Escaper SHELL_ESCAPE = Escapers.builder().addEscape('\"', "'").build();
    private static final Color FLASH_COLOR = new Color(255, 0, 0, 70);
    private static final int FLASH_DURATION = 2000;
    private final Client client;
    private final String appName;
    private final RuneLiteConfig runeLiteConfig;
    private final ClientUI clientUI;
    private final ScheduledExecutorService executorService;
    private final ChatMessageManager chatMessageManager;
    private final Path notifyIconPath;
    private final boolean terminalNotifierAvailable;
    private Instant flashStart;

    @Inject
    private Notifier(ClientUI clientUI, Client client2, RuneLiteConfig runeliteConfig, RuneLiteProperties runeLiteProperties, ScheduledExecutorService executorService, ChatMessageManager chatMessageManager) {
        this.client = client2;
        this.appName = runeLiteProperties.getTitle();
        this.clientUI = clientUI;
        this.runeLiteConfig = runeliteConfig;
        this.executorService = executorService;
        this.chatMessageManager = chatMessageManager;
        this.notifyIconPath = RuneLite.RUNELITE_DIR.toPath().resolve("icon.png");
        this.terminalNotifierAvailable = !Strings.isNullOrEmpty(RuneLiteProperties.getLauncherVersion()) && this.isTerminalNotifierAvailable();
        this.storeIcon();
    }

    public void notify(String message) {
        this.notify(message, TrayIcon.MessageType.NONE);
    }

    public void notify(String message, TrayIcon.MessageType type) {
        if (!this.runeLiteConfig.sendNotificationsWhenFocused() && this.clientUI.isFocused()) {
            return;
        }
        if (this.runeLiteConfig.requestFocusOnNotification()) {
            this.clientUI.requestFocus();
        }
        if (this.runeLiteConfig.enableTrayNotifications()) {
            this.sendNotification(this.appName, message, type);
        }
        if (this.runeLiteConfig.enableNotificationSound()) {
            Toolkit.getDefaultToolkit().beep();
        }
        if (this.runeLiteConfig.enableGameMessageNotification() && this.client.getGameState() == GameState.LOGGED_IN) {
            String formattedMessage = new ChatMessageBuilder().append(ChatColorType.HIGHLIGHT).append(message).build();
            this.chatMessageManager.queue(QueuedMessage.builder().type(ChatMessageType.CONSOLE).name(this.appName).runeLiteFormattedMessage(formattedMessage).build());
        }
        if (this.runeLiteConfig.enableFlashNotification()) {
            this.flashStart = Instant.now();
        }
    }

    public void processFlash(Graphics2D graphics) {
        if (this.flashStart == null || this.client.getGameCycle() % 40 >= 20) {
            return;
        }
        if (this.client.getGameState() != GameState.LOGGED_IN) {
            this.flashStart = null;
            return;
        }
        Color color = graphics.getColor();
        graphics.setColor(FLASH_COLOR);
        graphics.fill(new Rectangle(this.client.getCanvas().getSize()));
        graphics.setColor(color);
        if (Instant.now().minusMillis(2000L).isAfter(this.flashStart)) {
            this.flashStart = null;
        }
    }

    private void sendNotification(String title, String message, TrayIcon.MessageType type) {
        String escapedTitle = SHELL_ESCAPE.escape(title);
        String escapedMessage = SHELL_ESCAPE.escape(message);
        switch (OSType.getOSType()) {
            case Linux: {
                this.sendLinuxNotification(escapedTitle, escapedMessage, type);
                break;
            }
            case MacOS: {
                this.sendMacNotification(escapedTitle, escapedMessage);
                break;
            }
            default: {
                this.sendTrayNotification(title, message, type);
            }
        }
    }

    private void sendTrayNotification(String title, String message, TrayIcon.MessageType type) {
        if (this.clientUI.getTrayIcon() != null) {
            this.clientUI.getTrayIcon().displayMessage(title, message, type);
        }
    }

    private void sendLinuxNotification(String title, String message, TrayIcon.MessageType type) {
        ArrayList<String> commands = new ArrayList<String>();
        commands.add("notify-send");
        commands.add(title);
        commands.add(message);
        commands.add("-i");
        commands.add(SHELL_ESCAPE.escape(this.notifyIconPath.toAbsolutePath().toString()));
        commands.add("-u");
        commands.add(Notifier.toUrgency(type));
        commands.add("-t");
        commands.add(String.valueOf(10000));
        this.executorService.submit(() -> {
            try {
                Process notificationProcess = Notifier.sendCommand(commands);
                boolean exited = notificationProcess.waitFor(500L, TimeUnit.MILLISECONDS);
                if (exited && notificationProcess.exitValue() == 0) {
                    return;
                }
            }
            catch (IOException | InterruptedException ex) {
                log.debug("error sending notification", ex);
            }
            this.sendTrayNotification(title, message, type);
        });
    }

    private void sendMacNotification(String title, String message) {
        ArrayList<String> commands = new ArrayList<String>();
        if (this.terminalNotifierAvailable) {
            commands.add("terminal-notifier");
            commands.add("-group");
            commands.add("net.runelite.launcher");
            commands.add("-sender");
            commands.add("net.runelite.launcher");
            commands.add("-message");
            commands.add(DOUBLE_QUOTE + message + DOUBLE_QUOTE);
            commands.add("-title");
            commands.add(DOUBLE_QUOTE + title + DOUBLE_QUOTE);
        } else {
            commands.add("osascript");
            commands.add("-e");
            String script = "display notification \"" + message + "\" with title \"" + title + DOUBLE_QUOTE;
            commands.add(script);
        }
        try {
            Notifier.sendCommand(commands);
        }
        catch (IOException ex) {
            log.warn("error sending notification", ex);
        }
    }

    private static Process sendCommand(List<String> commands) throws IOException {
        return new ProcessBuilder(commands.toArray(new String[commands.size()])).redirectErrorStream(true).start();
    }

    private void storeIcon() {
        if (OSType.getOSType() == OSType.Linux && !Files.exists(this.notifyIconPath, new LinkOption[0])) {
            try (InputStream stream = Notifier.class.getResourceAsStream("/zenyte.png");){
                Files.copy(stream, this.notifyIconPath, new CopyOption[0]);
            }
            catch (IOException ex) {
                log.warn(null, ex);
            }
        }
    }

    private boolean isTerminalNotifierAvailable() {
        if (OSType.getOSType() == OSType.MacOS) {
            try {
                Process exec = Runtime.getRuntime().exec(new String[]{"terminal-notifier", "-help"});
                exec.waitFor();
                return exec.exitValue() == 0;
            }
            catch (IOException | InterruptedException e) {
                return false;
            }
        }
        return false;
    }

    private static String toUrgency(TrayIcon.MessageType type) {
        switch (type) {
            case WARNING: 
            case ERROR: {
                return "critical";
            }
        }
        return "normal";
    }
}

